#!/usr/bin/env python3

import sys
from validation import if_ipv4valide, if_macvalide  
from dhcp import dhcp_add, ip_other_mac_exists, mac_exists  
from config import get_dhcp_server, load_config  
from fabric import Connection
from paramiko.ssh_exception import NoValidConnectionsError, SSHException

def main():
    # vérifie qu'il y a exactement 2 arguments : mac et ip
    if len(sys.argv) != 3:
        print("usage: add-dhcp-client.py MAC IP")
        sys.exit(1)

    mac = sys.argv[1]
    ip = sys.argv[2]

    # valide l'ip, sinon quitte avec message d'erreur
    if not if_ipv4valide(ip):
        print("bad ip address.")
        sys.exit(1)

    # valide le mac, sinon quitte avec message d'erreur
    if not if_macvalide(mac):
        print("bad mac address.")
        sys.exit(1)

    # charge la config depuis le fichier yaml
    config = load_config('config_superviseur.yaml', True)
    # cherche quel serveur gère cette ip
    dhcp_server = get_dhcp_server(ip, config)

    if dhcp_server is None:
        print("unable to identify dhcp server")
        sys.exit(1)

    # récupère l'adresse ip du serveur
    dhcp_server_addr = list(dhcp_server.keys())[0]
    cnx = Connection(host=dhcp_server_addr, user='dhcp-mod')

    try:
        # si le mac n'existe pas déjà
        if not mac_exists(cnx, mac, config):
            # et si l'ip n'est pas déjà utilisée par un autre mac
            if not ip_other_mac_exists(cnx, ip, mac, config):
                dhcp_add(ip, mac, dhcp_server_addr, config)  # ajoute la ligne
            else:
                print("ip address already in use.")
                sys.exit(1)
        else:
            # si le mac existe, mais qu'on veut changer l'ip
            if not ip_other_mac_exists(cnx, ip, mac, config):
                dhcp_add(ip, mac, dhcp_server_addr, config)  # met à jour
            else:
                print("ip address already in use.")
                sys.exit(1)

    except NoValidConnectionsError:
        print('pas de connexion valide.')
    except SSHException as e:
        print(f'erreur de connexion ssh : {e}')


if __name__ == '__main__':
    main()
