#!/usr/bin/env python3

import sys
from dhcp import dhcp_list  
from config import get_dhcp_server, load_config
from fabric import Connection
from paramiko.ssh_exception import NoValidConnectionsError, SSHException

def main():
    # charge la config depuis le yaml
    config = load_config('config_superviseur.yaml', True)

    # si plus d'un paramètre en ligne de commande -> erreur
    if len(sys.argv) > 2:
        print("error: too much parameters")
        sys.exit(1)

    # si on a un paramètre (réseau ou ip de serveur)
    if len(sys.argv) == 2:
        param = sys.argv[1]

        # si le paramètre contient '/', c'est un réseau
        if '/' in param:
            found = False
            # on cherche le serveur qui gère ce réseau
            for server, network in config.get("dhcp-servers", {}).items():
                if network == param:
                    server_ip = server
                    found = True
                    break
            if not found:
                print("cannot identify dhcp server")
                sys.exit(1)
        else:
            # sinon c'est une ip serveur -> vérifie avec get_dhcp_server
            dhcp_server = get_dhcp_server(param, config)
            if not dhcp_server:
                print("cannot identify dhcp server")
                sys.exit(1)
            server_ip = list(dhcp_server.keys())[0]

        try:
            # récupère la liste des associations sur le serveur ciblé
            list_dhcp = dhcp_list(server_ip, config)
            mac_tot = {}
            ip_tot = {}

            # construit deux dictionnaires : un pour mac -> lignes, un pour ip -> lignes
            for addr in list_dhcp:
                mac = addr["mac"]
                ip = addr["ip"]
                line = f"dhcp-host={mac},{ip}"

                if mac not in mac_tot:
                    mac_tot[mac] = []
                mac_tot[mac].append(line)

                if ip not in ip_tot:
                    ip_tot[ip] = []
                ip_tot[ip].append(line)

            # affiche doublons mac
            for mac, lines in mac_tot.items():
                if len(lines) > 1:
                    print("duplicate mac addresses:")
                    for l in lines:
                        print(l)

            # affiche doublons ip
            for ip, lines in ip_tot.items():
                if len(lines) > 1:
                    print("duplicate ip addresses:")
                    for l in lines:
                        print(l)

        except NoValidConnectionsError:
            print('pas de connexion valide.')
        except SSHException as e:
            print(f'erreur ssh : {e}')

    else:
        # si aucun paramètre : vérifier tous les serveurs définis
        servers = config.get("dhcp-servers", {}).keys()
        for server_ip in servers:
            try:
                list_dhcp = dhcp_list(server_ip, config)
                mac_tot = {}
                ip_tot = {}

                for addr in list_dhcp:
                    mac = addr["mac"]
                    ip = addr["ip"]
                    line = f"dhcp-host={mac},{ip}"

                    if mac not in mac_tot:
                        mac_tot[mac] = []
                    mac_tot[mac].append(line)

                    if ip not in ip_tot:
                        ip_tot[ip] = []
                    ip_tot[ip].append(line)

                for mac, lines in mac_tot.items():
                    if len(lines) > 1:
                        print(f"{server_ip}: duplicate mac addresses:")
                        for l in lines:
                            print(l)

                for ip, lines in ip_tot.items():
                    if len(lines) > 1:
                        print(f"{server_ip}: duplicate ip addresses:")
                        for l in lines:
                            print(l)

            except NoValidConnectionsError:
                print(f'pas de connexion valide à {server_ip}')
            except SSHException as e:
                print(f'erreur ssh : {e}')


if __name__ == '__main__':
    main()
